#
#               MuPIF: Multi-Physics Integration Framework 
#                   Copyright (C) 2010 Borek Patzak
#
#       Czech Technical University, Faculty of Civil Engineering,
#       Department of Mechanics, 166 29 Prague, Czech Republic
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

from mupif import mesh

"""
It is important for the meshView to be able to represent a subset of master domain of a lower dimension,
eg. a surface in 3d mesh. As the cells in meshView do not have direct master cell in master mesh,
the meshView should be capable to maintain its own cells. 
On the other hand, if the surface is not directly represented in the master mesh (like an arbitrary cut),
the new mesh (and corresponding fields) are assumed to be created by some transfer operator,
creating a new mesh, as the direct mapping to master mesh does not exist.
"""

class MeshView(mesh.Mesh):
    """
    Mesh view represents a subset of master mesh, defined by mapping context.
    """
    def __init__(self, mappingContext, masterMesh):
        self.mapping     = mappingContext
        self.__masterMesh  = masterMesh
        
    def giveVertex(self, i):
        """Returns i-th vertex."""
        return self.__masterMesh.giveVertex(self.mapping.giveMasterVertexNumber(i))

    def giveCell(self, i):
        """Returns i-th cell."""
        return self.__masterMesh.giveCell(self.mapping.giveMasterCellNumber(i))
        
    def locateCell(self, position):
        """Locates cell containing given point"""
        masterCell = self.__masterMesh.locateCell(position)
        # now check if master cell is a part of local subset

        return masterCell

    def merge(self, mview):
        """Merges receiver with another mesh view instance,
        same underliying master mesh is required"""
        assert  self.__masterMesh != mview.__masterMesh, 'underlying meshes are different'
        self.mapping.merge (mview.mapping)
